/*****************************************************************************
* NwkApp.c
*
* The RF4CE controller node application
*
* Copyright (c) 2008, Freescale, Inc. All rights reserved.
*
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
*****************************************************************************/
#include "EmbeddedTypes.h"
#include "NwkInterface.h"
#include "TS_Interface.h"
#include "TMR_Interface.h"
#include "UartUtil.h"
#include "Uart_Interface.h"
#include "FunctionLib.h"
#include "PublicConst.h"
#include "keyboard.h"
#include "Display.h"
#include "Led.h"
#include "Sound.h"
#include "NwkApp.h"
#include "NwkAppInit.h"
#include "NV_Data.h"
#include "ApplicationConf.h"
#include "NVM_Interface.h"
#include "NwkDeviceType.h"
#include "NwkGlobals.h"
#include "PWR_Interface.h"
#include "FSLProfileConf.h"
#include "FSLProfileGlobals.h"
#include "ApplicationFunctionalityDefines.h"
#include "SPI_Interface.h"

/******************************************************************************
*******************************************************************************
* Private macros
*******************************************************************************
******************************************************************************/

/* Build the txOptions field to use when sending commands to paired devices */
#define gTxOptions_c                          ((gTxOptions_UseBroadcast_c         << 0) | \
                                               (gTxOptions_UseTargetLongAddress_c << 1) | \
                                               (gTxOptions_UseAck_c               << 2) | \
                                               (gTxOptions_UseSecurity_c          << 3) | \
                                               (gTxOptions_UseOneChannelOnly_c    << 4) | \
                                               (gTxOptions_UseChannelDesignator_c << 5) | \
                                               (gTxOptions_VendorSpecificData_c   << 6)) 
                              
/******************************************************************************
*******************************************************************************
* Private type definitions
*******************************************************************************
******************************************************************************/
typedef enum
{
  gAppStateIdle_c,
  gAppStateStart_c,
  gAppStateData_c,
  gAppStateUnpair_c,
  gAppStateFSLGetSupportedFeatures_c,
  gAppStateFSLFragTx_c,
  gAppStateFSLRmtPair_c,
  gAppStatePushButtonPairing_c,
  gAppStateZRCSendCommand_c
}appState_t;

typedef enum
{
  gAppSubStateStart_c,
  gAppSubStateWaitCnf_c,
  gAppSubStateEnd_c
}appSubState_t;

typedef struct appStateMachine_tag
{
   appState_t     state;    /* Store the current state */
   appSubState_t  subState; /* Store the current substate */
   uint8_t        deviceId; /* Store the desired deviceId for data request or unpair request */
#if gZRCProfileCommandTxRx_d  
   uint8_t        zrcCmdCode;
#endif   
}appStateMachine_t;


#if gFSLProfileRmtPairOrig_d
  /* Store info about paired devices (info necessary for remote pair process)*/
  typedef struct appPairedDeviceInfo_tag
  {
    appCapabilities_t        recipAppCapabilities;
    uint8_t                  recipDeviceTypeList[gMaxNrOfNodeDeviceTypes_c];
    uint8_t                  recipProfilesList[gMaxNrOfNodeProfiles_c];
  }appPairedDeviceInfo_t;
#endif /* gFSLProfileRmtPairOrig_d */

/******************************************************************************
*******************************************************************************
* Private Prototypes
*******************************************************************************
******************************************************************************/
static void*  App_HandleNlmeEvent(void);
static void*  App_HandleNldeEvent(void);
static void   App_HandleNlmeMessage(nwkNlmeToAppMsg_t* pMsgIn);
static void   App_HandleNldeMessage(nwkNldeToAppMsg_t* pMsgIn);

#if gFSLProfileTask_d 
  static void   App_FSLGetSupportedFeaturesState(event_t events, void* pMsgIn);
  static void*  App_HandleFSLProfileEvent(void);
  static void   App_HandleFSLProfileMessage(fslProfileToAppMsg_t* pMsgIn);
  static bool_t App_MessageIsForFSLProfile(nwkNldeToAppMsg_t* nwkNldeToAppMsg);
#endif /* gFSLProfileTask_d */

#if gZRCProfileCommandTxRx_d
  static void*  App_HandleZRCProfileEvent(void);
  static void   App_HandleZRCProfileMessage(zrcProfileToAppMsg_t* pMsgIn);
#endif /* gZRCProfileCommandTxRx_d */

#if gZRCProfileCommandTxRx_d
  static void   App_ZRCSendCommandState(event_t events, void* pMsgIn);
  static void   App_ZRCHandleCommandInd(zrcProfileCommandInd_t* pZRCProfileCommandInd);
#endif /* gZRCProfileCommandTxRx_d */


static void   App_PrintMenu(void);
static void   App_HandleUartInput(void);
static void   App_ExecuteUartRequestedAction(uint8_t lastKeyPressed, uint8_t deviceId);
static void   App_SetNodeMACAddress(void);
static void   App_UnpairResponse(void);
static void   App_HandleDataInd(nwkNldeDataInd_t* pNwkNldeDataInd);
static void   App_RxEnableRequest(uint8_t mode);
static void   App_PrintPairedDevices(void);
static void   App_UartRxCallBack(void);
static void   App_PrintResult(uint8_t status);

static void   App_StartState(event_t events, void* pMsgIn);
static void   App_DataState(event_t events, void* pMsgIn);
static void   App_UnpairState(event_t events, void* pMsgIn);

#if gFSLProfileFragTxOrig_d
  static void   App_FSLFragTxState(event_t events, void* pMsgIn);
  static void   App_FSLWriteDataInFragTxBuffer(void); 
#endif /* gFSLProfileFragTxOrig_d */

#if gFSLProfileRmtPairOrig_d
  static void   App_FSLRmtPairState(event_t events, void* pMsgIn);
#endif /* gFSLProfileRmtPairOrig_d */

#if gFSLProfilePollOrig_d
  static void   App_FSLPoll(uint8_t deviceId);
#endif /* gFSLProfilePollOrig_d */

#if gPBPOrig_d
  static void   App_PushButtonPairingState(event_t events, void* pMsgIn);
#endif /* gPBPOrig_d */

#ifdef PROCESSOR_MC1323X
  #if (gMC1323xMatrixKBD_d == TRUE)
    static void   App_HandleKeyboard(uint8_t events, uint8_t pressedKey);
  #else
    static void   App_HandleKeyboard(key_event_t events);
  #endif
#else
  static void   App_HandleKeyboard(key_event_t events);
#endif


/******************************************************************************
*******************************************************************************
* Private Memory Declarations
*******************************************************************************
******************************************************************************/
/* NLME message input queue */
static anchor_t               mNlmeAppInputQueue;  
/* NLDE message input queue */
static anchor_t               mNldeAppInputQueue; 

#if gFSLProfileTask_d
  /* FSL Profile message input queue */
  static anchor_t               mFSLProfileAppInputQueue; 
#endif /* gFSLProfileTask_d */

#if gZRCProfileCommandTxRx_d
  /* ZRC Profile message input queue */
  static anchor_t               mZRCProfileAppInputQueue; 
#endif /* gZRCProfileCommandTxRx_d */

#if gPBPTask_d
/* Push Button message input queue */
static anchor_t               mPushButtonAppInputQueue; 
#endif

/* Stores the application information */
static appStateMachine_t      appStateMachine;

#if gFSLProfileRmtPairOrig_d
  /* Array that keeps application info about paired devices */
  static appPairedDeviceInfo_t appPairedDeviceInfo[gMaxPairingTableEntries_c];
#endif /* gFSLProfileRmtPairOrig_d */

#if gFSLProfilePollOrig_d
  /* Stores poll information */
  static bool_t bPollEnable[gMaxPairingTableEntries_c];
#endif /* gFSLProfilePollOrig_d */

/******************************************************************************
*******************************************************************************
* Public memory declarations
*******************************************************************************
******************************************************************************/
/* This data set contains network layer variables to be preserved across resets */
  /* Application info about paired devices must be saved in Flash */ 
  NvDataItemDescription_t const gaNvAppDataSet[] = 
  {
#if gFSLProfileRmtPairOrig_d
    {&appPairedDeviceInfo[0], sizeof(appPairedDeviceInfo_t) * gMaxPairingTableEntries_c},
#endif /* gFSLProfileRmtPairOrig_d */
    {NULL, 0}       /* Required end-of-table marker. */
  };

/* Application capabilities field for this node */
appCapabilities_t localAppCapabilities = 
                              {
                              gAppCapabilities_UseUserString_c,
                              gAppCapabilities_NrOfSupportedDeviceTypes_c,
                              0x00,  /* reserved1 */
                              gAppCapabilities_NrOfSupportedProfiles_c,
                              0x00   /* reserved2 */
                              };
                                                       
/* List of the device types supported by this node */                                                     
uint8_t  localDeviceTypesList[] = 
                              {
                              gSupportedDeviceType1_c,
                              gSupportedDeviceType2_c,
                              gSupportedDeviceType3_c
                              };
                                          
/* List of the profiles supported by this node */                                                     
uint8_t  localProfilesList[] = 
                              {
                              gSupportedProfile1_c,
                              gSupportedProfile2_c,
                              gSupportedProfile3_c,
                              gSupportedProfile4_c,
                              gSupportedProfile5_c,
                              gSupportedProfile6_c,
                              gSupportedProfile7_c
                              };
  
/* Definition of the RF4CE device type names */
const uint8_t                 deviceType1[]           = "Remote Control"; 
const uint8_t                 deviceType2[]           = "Television"; 
const uint8_t                 deviceType3[]           = "Projector"; 
const uint8_t                 deviceType4[]           = "Player"; 
const uint8_t                 deviceType5[]           = "Recorder"; 
const uint8_t                 deviceType6[]           = "Video player/recorder"; 
const uint8_t                 deviceType7[]           = "Audio player/recorder"; 
const uint8_t                 deviceType8[]           = "Audio video recorder"; 
const uint8_t                 deviceType9[]           = "Set top box"; 
const uint8_t                 deviceType10[]          = "Home theater system"; 
const uint8_t                 deviceType11[]          = "Media center/PC"; 
const uint8_t                 deviceType12[]          = "Game console"; 
const uint8_t                 deviceType13[]          = "Satellite radio receiver"; 
const uint8_t                 deviceType14[]          = "IR extender"; 
const uint8_t                 deviceType15[]          = "Monitor"; 
const uint8_t                 deviceType16[]          = "Generic"; 


/* Pointer array refering the RF4CE device type names */
const uint8_t* strDeviceTypeName[]=
{
  NULL,
  deviceType1,
  deviceType2,
  deviceType3,
  deviceType4,
  deviceType5,
  deviceType6,
  deviceType7,
  deviceType8,
  deviceType9,
  deviceType10,
  deviceType11,
  deviceType12,
  deviceType13,
  deviceType14,
  deviceType15,
  deviceType16,
};


/******************************************************************************
*******************************************************************************
* Public Functions
*******************************************************************************
******************************************************************************/

/*****************************************************************************
* App_Init
*
* Initializes the application
*
* Initialization function for the App Task. This is called during
* initialization and should contain any application specific initialization
* (ie. hardware initialization/setup, table initialization, power up
* notificaiton.
*****************************************************************************/
void App_Init(void)
{
  /* Initialize LED driver */
  LED_Init();
  /* Register the keyboard input */
  KBD_Init(App_HandleKeyboard);
  /* Initialize SPI */
  SPI_Init();
  /* Initialize LCD (NCB/AXIOM only) */
  LCD_Init(); 
  /* Intialize UART module */
  Uart_ModuleInit();
  UartX_SetBaud(gUartDefaultBaud_c);
  UartX_SetRxCallBack(App_UartRxCallBack);
  /* Initialize the LPM module */
  PWRLib_Init();
  /* Enable all interrupts */  
  IrqControlLib_EnableAllIrqs();
  /* Restore data from Flash */
  NvRestoreDataSet(gNvDataSet_App_ID_c);
  
  /* Do not allow device to enter low power mode */
  PWR_DisallowDeviceToSleep();
  /* Initialize the queues used to store messages received trough the 2 RF4CE SAP */
  MSG_InitQueue(&mNlmeAppInputQueue);
  MSG_InitQueue(&mNldeAppInputQueue);

#if gFSLProfileTask_d
  /* Initialize the queue used to store messages received trough the FSL Profile SAP */
  MSG_InitQueue(&mFSLProfileAppInputQueue);
#endif /* gFSLProfileTask_d */

#if gZRCProfileCommandTxRx_d
  /* Initialize the queue used to store messages received trough the FSL Profile SAP */
  MSG_InitQueue(&mZRCProfileAppInputQueue);
#endif /* gZRCProfileCommandTxRx_d */

#if gPBPTask_d
  /* Initialize the queue used to store messages received trough the Push Button SAP */
  MSG_InitQueue(&mPushButtonAppInputQueue);  
#endif


  /* Reset NWK layer */
  (void)NLME_ResetRequest(FALSE);
  /* Set the node's MAC address */
  App_SetNodeMACAddress();
  /* Print a welcome message to the UART */
  UartUtil_Print("\n\rRF4CE Controller Node demo application is initialized and ready.\n\r", gAllowToBlock_d);  
  /* Print a welcome message to the LCD also */
  LCD_ClearDisplay();  
  LCD_WriteString(1,"Controller Node");
  LCD_WriteString(2,"initialized&ready"); 
  /* Turn off all leds */
  LED_TurnOffAllLeds();
  
  /* Init FSL profile procedures */
#if gFSLProfileFragTxOrig_d
  FSLProfile_InitFragTxOrigProcedure();
#endif /* gFSLProfileFragTxOrig_d */

#if gFSLProfileFragTxRecip_d
  FSLProfile_InitFragTxRecipProcedure();
#endif /* gFSLProfileFragTxRecip_d */

#if gFSLProfilePollOrig_d
  FLib_MemSet((void*)&bPollEnable[0], TRUE, gMaxPairingTableEntries_c);
  FSLProfile_InitPollOrigProcedure();
  /* Configure poll request */
  FSLProfile_PollConfigRequest(gAppFSLPollInterval_c, gAppFSLPollRxOnInterval_c);
#endif /* gFSLProfilePollOrig_d */
  
#if gFSLProfileRmtPairOrig_d
  FSLProfile_InitRmtPairOrigProcedure();
#endif /* gFSLProfileRmtPairOrig_d */
  
  /* Init PBP procedure */
#if gPBPOrig_d
  PBP_InitPushButtonPairOrig();
#endif /* gPBPOrig_d */

#if gZRCProfileCommandTxRx_d
  ZRCProfile_InitCommandTxRx();  
#endif /* gZRCProfileCommandTxRx_d */

  /* Start the node */
  appStateMachine.deviceId  = gInvalidDeviceId_c;
  appStateMachine.state     = gAppStateStart_c;
  appStateMachine.subState  = gAppSubStateStart_c;
  TS_SendEvent(gAppTaskID, gAppEvtStateStart_c); 
}


/*****************************************************************************
*  App_MainTask
*
*  Application task. Responds to events for the application.
*****************************************************************************/
void App_MainTask(event_t events)
{
  void* pMsgIn = NULL;
  
  if(events & gAppEvtStateEnd_c)
  {
    /* Complete the current state */
    appStateMachine.state    = gAppStateIdle_c;
    appStateMachine.subState = gAppSubStateStart_c;
    appStateMachine.deviceId = gInvalidDeviceId_c;
  }
  
  /* Treat NLME input if received */     
  if (events & gAppEvtMsgFromNlme_c) 
    pMsgIn = App_HandleNlmeEvent();
  /* Treat NLDE input if received */     
  else if (events & gAppEvtMsgFromNlde_c) 
    pMsgIn = App_HandleNldeEvent();
#if gPBPTask_d 
  else if (events & gAppEvtMsgFromPushButton_c) 
    pMsgIn = MSG_DeQueue(&mPushButtonAppInputQueue);
#endif  
#if gFSLProfileTask_d
  /* Treat FSL profile input if received */    
  else if (events & gAppEvtMsgFromFSLProfile_c) 
    pMsgIn = App_HandleFSLProfileEvent();  
#endif /* gFSLProfileTask_d */
#if gZRCProfileCommandTxRx_d
  /* Treat ZRC profile input if received */
  else if (events & gAppEvtMsgFromZRCProfile_c) 
    pMsgIn = App_HandleZRCProfileEvent();  
#endif /* gZRCProfileCommandTxRx_d */

  
  /* Handle the event and the received message into the state */ 
  switch(appStateMachine.state)
  {
    case gAppStateStart_c:
      App_StartState(events, pMsgIn);
      break;
                  
    case gAppStateData_c:
      App_DataState(events, pMsgIn);
      break;
    
    case gAppStateUnpair_c:
      App_UnpairState(events, pMsgIn);
      break;
           
#if gFSLProfileTask_d
    case gAppStateFSLGetSupportedFeatures_c:
      App_FSLGetSupportedFeaturesState(events, pMsgIn);
      break;
#endif /* gFSLProfileTask_d */

#if gFSLProfileFragTxOrig_d
    case gAppStateFSLFragTx_c:
      App_FSLFragTxState(events, pMsgIn);
      break;
#endif /* gFSLProfileFragTxOrig_d */
                
#if gFSLProfileRmtPairOrig_d
    case gAppStateFSLRmtPair_c:
      App_FSLRmtPairState(events, pMsgIn);
      break;
#endif /* gFSLProfileRmtPairOrig_d */
            
#if gPBPTask_d
    case gAppStatePushButtonPairing_c:
      App_PushButtonPairingState(events, pMsgIn);
      break;
#endif /* gPBPTask_d */
   
#if gZRCProfileCommandTxRx_d
    case gAppStateZRCSendCommand_c:
      App_ZRCSendCommandState(events, pMsgIn);
      break;
#endif /* gZRCProfileCommandTxRx_d */
           
    case gAppStateIdle_c:
      /* Print menu if requested */    
      if (events & gAppEvtPrintMenu_c) 
        App_PrintMenu();
      /* Treat UART input if received */     
      if (events & gAppEvtRxFromUart_c) 
        App_HandleUartInput();
      break;

    default:
      break;            
  }
  
   /* Free the received message if this is not null */
  if(pMsgIn)
    MSG_Free(pMsgIn);
  
  /* Check for pending messages in any of queues */ 
  if(MSG_Pending(&mNlmeAppInputQueue))
    TS_SendEvent(gAppTaskID, gAppEvtMsgFromNlme_c);
    
  if(MSG_Pending(&mNldeAppInputQueue))
    TS_SendEvent(gAppTaskID, gAppEvtMsgFromNlde_c); 
  
#if gPBPTask_d 
  if(MSG_Pending(&mPushButtonAppInputQueue))
    TS_SendEvent(gAppTaskID, gAppEvtMsgFromPushButton_c);    
#endif

#if gFSLProfileTask_d
  if(MSG_Pending(&mFSLProfileAppInputQueue))
    TS_SendEvent(gAppTaskID, gAppEvtMsgFromFSLProfile_c);
#endif /* gFSLProfileTask_d */

#if gZRCProfileCommandTxRx_d
  if(MSG_Pending(&mZRCProfileAppInputQueue))
    TS_SendEvent(gAppTaskID, gAppEvtMsgFromZRCProfile_c);
#endif /* gZRCProfileCommandTxRx_d */
}

/******************************************************************************
*******************************************************************************
* Private Functions
*******************************************************************************
******************************************************************************/


/*****************************************************************************
*  App_HandleNlmeEvent
*
*  Handles messages received on the NLME SAP
*
*****************************************************************************/
static void* App_HandleNlmeEvent(void)
{
  void *pMsgIn;
  /* Try to get the first message in the Nlme message queue */  
  pMsgIn = MSG_DeQueue(&mNlmeAppInputQueue);
  
  if((pMsgIn) && (appStateMachine.state == gAppStateIdle_c))
  {
    /* Handle the message in Idle state */
    App_HandleNlmeMessage((nwkNlmeToAppMsg_t*)pMsgIn); 
    MSG_Free(pMsgIn);
    pMsgIn = NULL;
  }
  return pMsgIn;
}

/*****************************************************************************
*  App_HandleNldeEvent
*
*  Handles messages received on the NLDE SAP
*
*****************************************************************************/
static void* App_HandleNldeEvent(void)
{
  void *pMsgIn;
  /* Try to get the first message in the Nlde message queue */
  pMsgIn = MSG_DeQueue(&mNldeAppInputQueue);        
  if((pMsgIn) && (appStateMachine.state == gAppStateIdle_c))
  {
    /* Handle the message in Idle state */
    App_HandleNldeMessage((nwkNldeToAppMsg_t*)pMsgIn); 
    MSG_Free(pMsgIn);
    pMsgIn = NULL;
  }
  return pMsgIn;
}



#if gFSLProfileTask_d
/*****************************************************************************
*  App_HandleFSLProfileEvent
*
*  Handles messages received on the FSL Profile SAP
*
*****************************************************************************/
static void* App_HandleFSLProfileEvent(void)
{
  void *pMsgIn;
  /* Try to get the first message from the FSL profile message queue */
  pMsgIn = MSG_DeQueue(&mFSLProfileAppInputQueue);        
  if((pMsgIn) && (appStateMachine.state == gAppStateIdle_c))
  {
    /* Handle the message in Idle state */
    App_HandleFSLProfileMessage((fslProfileToAppMsg_t*)pMsgIn); 
    MSG_Free(pMsgIn);
    pMsgIn = NULL;
  }
  return pMsgIn;
}
#endif /* gFSLProfileTask_d */


#if gZRCProfileCommandTxRx_d
/*****************************************************************************
*  App_HandleZRCProfileEvent
*
*  Handles messages received on the FSL Profile SAP
*
*****************************************************************************/
static void* App_HandleZRCProfileEvent(void)
{
  void *pMsgIn;
  /* Try to get the first message from the ZRC profile message queue */
  pMsgIn = MSG_DeQueue(&mZRCProfileAppInputQueue);        
  if((pMsgIn) && (appStateMachine.state == gAppStateIdle_c))
  {
    /* Handle the message in Idle state */
    App_HandleZRCProfileMessage((zrcProfileToAppMsg_t*)pMsgIn); 
    MSG_Free(pMsgIn);
    pMsgIn = NULL;
  }
  return pMsgIn;
}
#endif /* gZRCProfileCommandTxRx_d */


/*****************************************************************************
*  App_HandleNlmeMessage
*
*  Handles the messages received from NWK layer
*
*****************************************************************************/
static void App_HandleNlmeMessage(nwkNlmeToAppMsg_t* pMsgIn)
{
  
  switch(pMsgIn->msgType) 
  {
    /* Unpair indication received */
    case gNwkNlmeUnpairInd_c:
      /* Save the device identifier for use it in unpair response, if requested by the user */
      appStateMachine.deviceId = pMsgIn->msgData.nwkNlmeUnpairInd.deviceId;
        
      UartUtil_Print("\n\rReceived Unpair Request from device ", gAllowToBlock_d); 
      UartUtil_Print(nodeData.pairTableEntry[appStateMachine.deviceId].recipUserString, gAllowToBlock_d); 
      UartUtil_Print("\n\rPress 'Y' key to delete the device from the pair table", gAllowToBlock_d); 
      break;
          
    default:
      break;          
  }
}


/*****************************************************************************
*  App_HandleNldeMessage
*
*  Handles the messages received from NWK NLDE layer
*
*****************************************************************************/
static void App_HandleNldeMessage(nwkNldeToAppMsg_t* pMsgIn)
{
  switch(pMsgIn->msgType) 
  {
    /* Data indication received */
    case gNwkNldeDataInd_c: 
      App_HandleDataInd(&pMsgIn->msgData.nwkNldeDataInd); 
      break;
      
    default:
      break;          
  }
}


#if gFSLProfileTask_d
/*****************************************************************************
*  App_HandleFSLProfileMessage
*
*  Handles the messages received from FSL Profile layer
*
*****************************************************************************/
static void App_HandleFSLProfileMessage(fslProfileToAppMsg_t* pMsgIn)
{
  switch(pMsgIn->msgType) 
  {
#if gFSLProfileFragTxRecip_d
    /* Fragmented data start indication received */
    case gFSLProfileFragStartInd_c: 
      UartUtil_Print("\n\rFragTx start (from device ", gAllowToBlock_d);
      UartUtil_Print(nodeData.pairTableEntry[pMsgIn->msgData.fslProfileFragStartInd.deviceId].recipUserString, gAllowToBlock_d);
      UartUtil_Print("). Data length = 0x", gAllowToBlock_d);
      UartUtil_PrintHex((uint8_t*)&pMsgIn->msgData.fslProfileFragStartInd.fragDataLen, 2, 1);
      UartUtil_Print(".\n\r", gAllowToBlock_d);
      break;
    
    /* Fragmented data indication received */
    case gFSLProfileFragInd_c:
      if(pMsgIn->msgData.fslProfileFragInd.status != gNWSuccess_c)
      {
        UartUtil_Print("\n\rFragTx FAILED!", gAllowToBlock_d);
      }
      else
      {
        UartUtil_Print("\n\rFragTx end (from device ", gAllowToBlock_d);
        UartUtil_Print(nodeData.pairTableEntry[pMsgIn->msgData.fslProfileFragInd.deviceId].recipUserString, gAllowToBlock_d);
        UartUtil_Print("). Data length = 0x", gAllowToBlock_d);
        UartUtil_PrintHex((uint8_t*)&pMsgIn->msgData.fslProfileFragInd.fragDataLen, 2, 1);
        UartUtil_Print(".\n\r", gAllowToBlock_d);
        
        UartUtil_Print("\n\rSet FragTxRx buffer state to Free... ", gAllowToBlock_d); 
        App_PrintResult(FSLProfile_SetFragTxRxBufferStateRequest(gFragTxRxBufferFree_c));
      }
      break;
#endif /* gFSLProfileFragTxRecip_d */
      
#if gFSLProfilePollOrig_d
    /* Poll event received */
    case gFSLProfilePollEvent_c:
      UartUtil_Print("\n\rDevice ", gAllowToBlock_d);
      UartUtil_Print(nodeData.pairTableEntry[pMsgIn->msgData.fslProfilePollEvent.deviceId].recipUserString, gAllowToBlock_d);
      if(pMsgIn->msgData.fslProfilePollEvent.bDataAvailable)
        UartUtil_Print(" has poll data.\n\r", gAllowToBlock_d);
      else
        UartUtil_Print(" has no poll data.\n\r", gAllowToBlock_d);
      break;
     
    /* Poll confirm received */
    case gFSLProfilePollCnf_c:
      if(pMsgIn->msgData.fslProfilePollCnf.status != gNWAborted_c)
      {       
        UartUtil_Print("\n\rDevice ", gAllowToBlock_d);
        UartUtil_Print(nodeData.pairTableEntry[pMsgIn->msgData.fslProfilePollEvent.deviceId].recipUserString, gAllowToBlock_d);
        UartUtil_Print(" can not be polled.\n\r", gAllowToBlock_d);
      }
      else
      {
        UartUtil_Print("\n\rPoll was stopped.\n\r", gAllowToBlock_d); 
      }
      break;
#endif /* gFSLProfilePollOrig_d */

    default:
      break;          
  }
}
#endif /* gFSLProfileTask_d */



#if gZRCProfileCommandTxRx_d
/*****************************************************************************
*  App_HandleZRCProfileMessage
*
*  Handles the messages received from ZRC Profile layer
*
*****************************************************************************/
static void App_HandleZRCProfileMessage(zrcProfileToAppMsg_t* pMsgIn)
{
  switch(pMsgIn->msgType) 
  {

#if gZRCProfileCommandTxRx_d
    /* ZRC command indication received */
    case gZRCProfileCommandInd_c: 
      App_ZRCHandleCommandInd(&pMsgIn->msgData.zrcProfileCommandInd);
      break;
#endif /* gZRCProfileCommandTxRx_d */
      
    default:
      break;          
  }
}
#endif /* gZRCProfileCommandTxRx_d */

/*****************************************************************************
*  App_PrintMenu
*
*  Used for printing a menu on a serial Hyperterminal
*
*****************************************************************************/
static void App_PrintMenu(void)
{
  UartUtil_Print("\n\r-----Remote Controller's MENU -----", gAllowToBlock_d); 
  UartUtil_Print("\n\rPlease choose an option:\n\r", gAllowToBlock_d); 
  UartUtil_Print("  t. Transmit data to a paired device >>\n\r", gAllowToBlock_d);                                    
  UartUtil_Print("  u. Unpair a paired device >>\n\r", gAllowToBlock_d);            
#if gFSLProfileTask_d
  UartUtil_Print("  g. Get the FSL private profile features supported by a paired device >>\n\r", gAllowToBlock_d);            
#endif /* gFSLProfileTask_d */
#if gFSLProfileFragTxOrig_d
  UartUtil_Print("  a. Transmit FragTx data to a paired device >>\n\r", gAllowToBlock_d);            
#endif /* gFSLProfileFragTxOrig_d */
#if gFSLProfilePollOrig_d
  UartUtil_Print("  p. Poll a paired device >>\n\r", gAllowToBlock_d);            
#endif /* gFSLProfilePollOrig_d */
#if gFSLProfileRmtPairOrig_d
  UartUtil_Print("  m. Pair remote first two paired devices\n\r", gAllowToBlock_d);             
#endif /* gFSLProfileRmtPairOrig_d */
#if gPBPOrig_d
  UartUtil_Print("  b. Push button pairing\n\r", gAllowToBlock_d);                                    
#endif /* gPBPOrig_d */
#if gZRCProfileCommandTxRx_d
  UartUtil_Print("  c. Send a ZRC command to a paired device >>\n\r", gAllowToBlock_d);  
  UartUtil_Print("  d. Send a ZRC Discovery command to a paired device >>\n\r", gAllowToBlock_d);   
#endif /* gZRCProfileCommandTxRx_d */
  UartUtil_Print("  l. List paired devices\n\r", gAllowToBlock_d);
  UartUtil_Print("  z. Set Receiver Mode >>\n\r", gAllowToBlock_d);  
  UartUtil_Print("  w. Allow device to enter low power mode\n\r", gAllowToBlock_d);  
  UartUtil_Print("  n. Do not allow device to enter low power mode\n\r", gAllowToBlock_d);  
  UartUtil_Print("  r. Refresh menu\n\r", gAllowToBlock_d); 
  /* Print menu on the LCD */
  LCD_ClearDisplay();  
  LCD_WriteString(1,"To see menu use");
  LCD_WriteString(2,"Hyperterminal");             
}

/*****************************************************************************
*  App_HandleUartInput
*
*  Handles data received on the UART
*
*****************************************************************************/
static void App_HandleUartInput(void)
{
  static uint8_t  oldMenuKey  = 0xFF;
  uint8_t         pressedKey = 0;
  
  /* Get byte from UART */
  while(UartX_GetByteFromRxBuffer(&pressedKey)) 
  {
    if(pressedKey >= '1' && pressedKey <= '9')
    {
      App_ExecuteUartRequestedAction(oldMenuKey, pressedKey - '1');
    }
    
    switch(pressedKey)
    {        
      case 'l': 
      case 'L': 
        UartUtil_Print("\n\rList of paired devices:\n\r", gAllowToBlock_d); 
        App_PrintPairedDevices();
        break;
       
      case 't': 
      case 'T': 
        UartUtil_Print("\n\rPlease select a device to transmit a data request:\n\r", gAllowToBlock_d); 
        LCD_ClearDisplay();  
        LCD_WriteString(1,"Command request");
        App_PrintPairedDevices();
        break;                   
       
      case 'u':         
      case 'U':         
        UartUtil_Print("\n\rPlease select a device to unpair:\n\r", gAllowToBlock_d); 
        LCD_ClearDisplay();  
        LCD_WriteString(1,"Delete device.");
        App_PrintPairedDevices();
        break;
        
#if gFSLProfileTask_d
      case 'g': 
      case 'G': 
        UartUtil_Print("\n\rPlease select a device to transmit a GetSupportedFeatures request:\n\r", gAllowToBlock_d); 
        LCD_ClearDisplay();  
        LCD_WriteString(1,"GetSupportedFeatures request");
        App_PrintPairedDevices();
        break;
#endif /* gFSLProfileTask_d */

#if gFSLProfileFragTxOrig_d
      case 'a': 
      case 'A': 
        UartUtil_Print("\n\rPlease select a device to transmit a FragTx request:\n\r", gAllowToBlock_d); 
        LCD_ClearDisplay();  
        LCD_WriteString(1,"FragTx request");
        App_PrintPairedDevices();
        break;
#endif /* gFSLProfileFragTxOrig_d */

#if gFSLProfilePollOrig_d
      case 'p':
      case 'P':
        UartUtil_Print("\n\rPlease select a device to be polled:\n\r", gAllowToBlock_d); 
        LCD_ClearDisplay();  
        LCD_WriteString(1,"Poll device");
        App_PrintPairedDevices();
        break;
#endif /* gFSLProfilePollOrig_d */
         
#if gFSLProfileRmtPairOrig_d
      case 'm':
      case 'M':
        /* Begin remote pair state */
        appStateMachine.state = gAppStateFSLRmtPair_c; 
        TS_SendEvent(gAppTaskID, gAppEvtStateStart_c);  
        break;
#endif /* gFSLProfileRmtPairOrig_d */
        
#if gPBPOrig_d
      case 'b':
      case 'B':
        /* Begin push data pairing state */
        appStateMachine.state = gAppStatePushButtonPairing_c; 
        TS_SendEvent(gAppTaskID, gAppEvtStateStart_c); 
        break;
#endif /* gPBPOrig_d */

#if gZRCProfileCommandTxRx_d
      case 'c':
      case 'C':
        UartUtil_Print("\n\rPlease select a device to transmit a ZRC command request:\n\r", gAllowToBlock_d); 
        LCD_ClearDisplay();  
        LCD_WriteString(1,"ZRC Command request");
        App_PrintPairedDevices();
        break;
        
      case 'd': 
      case 'D': 
        UartUtil_Print("\n\rPlease select a device to transmit a ZRC discovery request:\n\r", gAllowToBlock_d); 
        LCD_ClearDisplay();  
        LCD_WriteString(1,"Discovery req");
        App_PrintPairedDevices();
        break;
#endif /* gZRCProfileCommandTxRx_d */
            
      case 'r':
      case 'R':
        TS_SendEvent(gAppTaskID, gAppEvtPrintMenu_c);    
        break;  
        
      case 'y':
      case 'Y':
        App_UnpairResponse();
        break;
          
      case 'z':
      case 'Z':
        UartUtil_Print("\n\rPlease select an option:", gAllowToBlock_d); 
        UartUtil_Print("\n\r1. Rx On", gAllowToBlock_d); 
        UartUtil_Print("\n\r2. Rx Off", gAllowToBlock_d);
        UartUtil_Print("\n\r3. Intermittent Rx", gAllowToBlock_d);  
        break;                   

      case 'w':
      case 'W':
        UartUtil_Print("\n\rEntering in low power mode is allowed by the application.",gAllowToBlock_d);
        /* Allow device to enter the low power mode. The low power mode is configured in PWR_Configuration.h */
        PWR_AllowDeviceToSleep();
        break;

      case 'n':
      case 'N':
        UartUtil_Print("\n\rEntering in low power mode is not allowed by the application.",gAllowToBlock_d);
        /* Do not allow device to enter the low power mode. The low power mode is configured in PWR_Configuration.h */
        PWR_DisallowDeviceToSleep();
        break;
      default:
        break;          
    }
    oldMenuKey = pressedKey;
  }
} 


/*****************************************************************************
*  App_ExecuteUartRequestedAction
*
*  Depending of the last key pressed and the new key pressed,
*  different functions are called
*
*****************************************************************************/
static void App_ExecuteUartRequestedAction(uint8_t lastKeyPressed, uint8_t deviceId)
{
  switch(lastKeyPressed)
  {
    case 't':
    case 'T':
      /* Begin data state */
      appStateMachine.state     = gAppStateData_c; 
      TS_SendEvent(gAppTaskID, gAppEvtStateStart_c);  
      appStateMachine.deviceId  = deviceId;
      break;
      
    case 'u':
    case 'U':
      /* Begin unpair state */
      appStateMachine.state     = gAppStateUnpair_c; 
      TS_SendEvent(gAppTaskID, gAppEvtStateStart_c);  
      appStateMachine.deviceId  = deviceId;		   
      break;

#if gFSLProfileTask_d
    case 'g':
    case 'G':
      /* Begin GetSupportedFeatures state */
      appStateMachine.state     = gAppStateFSLGetSupportedFeatures_c; 
      TS_SendEvent(gAppTaskID, gAppEvtStateStart_c);  
      appStateMachine.deviceId  = deviceId;		   
      break;
#endif /* gFSLProfileTask_d */
      
#if gFSLProfileFragTxOrig_d
    case 'a':
    case 'A':
      /* Begin fragTx state */
      appStateMachine.state     = gAppStateFSLFragTx_c; 
      TS_SendEvent(gAppTaskID, gAppEvtStateStart_c);  
      appStateMachine.deviceId  = deviceId;		   
      break;
#endif /* gFSLProfileFragTxOrig_d */
    
#if gFSLProfilePollOrig_d
    case 'p':
    case 'P':
      /* Poll or stop polling the requested device */
      App_FSLPoll(deviceId);		   
      break;
#endif /* gFSLProfilePollOrig_d */
      
#if gZRCProfileCommandTxRx_d
    case 'c':
    case 'C':
      /* Begin data state */
      appStateMachine.state     = gAppStateZRCSendCommand_c; 
      TS_SendEvent(gAppTaskID, gAppEvtStateStart_c);  
      appStateMachine.deviceId  = deviceId;
      appStateMachine.zrcCmdCode =  gAppZRCProfileCmdCode_c;
      break;  

    case 'd':
    case 'D':
      /* Begin data state */
      appStateMachine.state     = gAppStateZRCSendCommand_c; 
      TS_SendEvent(gAppTaskID, gAppEvtStateStart_c);  
      appStateMachine.deviceId  = deviceId;
      appStateMachine.zrcCmdCode =  gZRC_CmdCode_DiscoveryRequest_c;
      break;                        
#endif /* gZRCProfileCommandTxRx_d */
      
    case 'z':
    case 'Z':
      App_RxEnableRequest(deviceId);
      break;
    
    default:
      break;                                                                          
  }
}   


/*****************************************************************************
*  App_SetNodeMACAddress - Sets the MAC address of the node
*
*****************************************************************************/
static void App_SetNodeMACAddress(void)
{
  /* MAC address of the controller node */
  uint8_t localMACaddress[] = {gDefaultValueOfExtendedAddress_c};
  /* Initialize MAC address of the board */
  NWK_SetMacAddress(localMACaddress);
}

/*****************************************************************************
*  App_StartState - Starts the node
*
*****************************************************************************/
static void App_StartState(event_t events, void* pMsgIn)
{
  uint8_t status;
  
  if((events & gAppEvtStateStart_c) && (appStateMachine.subState == gAppSubStateStart_c))
  { 
    /* Print an info message to the UART */
    UartUtil_Print("\n\rStart node as controller... ", gAllowToBlock_d);  
    LCD_ClearDisplay();  
    LCD_WriteString(1," Start node");
    LCD_WriteString(2,"as controller");              
    /* Try to start the node as controller */
    status = NLME_StartRequest();
    /* Exit the state if this is not successful, otherwise wait confirm */
    if(gNWSuccess_c == status)
    {
      appStateMachine.subState = gAppSubStateWaitCnf_c;
    }
    else
    {
      appStateMachine.subState = gAppSubStateEnd_c;
    }
  }
  
  switch(appStateMachine.subState)
  {
    case gAppSubStateWaitCnf_c:
      if((events & gAppEvtMsgFromNlme_c) && 
         (pMsgIn != NULL))
      {
        nwkNlmeToAppMsg_t* pNlmeMsgIn = (nwkNlmeToAppMsg_t*)pMsgIn;
        
        /* Start confirm received */
        if(pNlmeMsgIn->msgType == gNwkNlmeStartCnf_c)
        {
          status = pNlmeMsgIn->msgData.nwkNlmeStartCnf.status;
          if(status == gNWSuccess_c)
          {
            uint8_t   userString[] = gUserString_c;
            
            /* Set nwkUserString NIB */
            status = NLME_SetRequest(gNwkNib_UserString_c, 0x00, userString);
          }
          appStateMachine.subState = gAppSubStateEnd_c;
        }
      }
      break;
    default:
      break;        
  }
  
  if(appStateMachine.subState == gAppSubStateEnd_c)
  {
    /* Print the status */
    App_PrintResult(status);
    
    /* Send event to end the state */
    TS_SendEvent(gAppTaskID, gAppEvtStateEnd_c);
    /* If successful, send event for printing the menu */
    if(status == gNWSuccess_c)
      TS_SendEvent(gAppTaskID, gAppEvtPrintMenu_c);     
    else
      App_Init();
  }     
       
}

/*****************************************************************************
*  App_DataState
*
*  Sends a data request specific with the current selected profileId.
*
*****************************************************************************/
static void App_DataState(event_t events, void* pMsgIn)
{
  uint8_t status;

  if((events & gAppEvtStateStart_c) && (appStateMachine.subState == gAppSubStateStart_c))
  {
    uint8_t appCmdPayload[]   = gAppCmdPayload_c; 
    uint8_t vendorId[]        = gDefaultValueOfVendorId_c;

    UartUtil_Print("\n\rSending data... ", gAllowToBlock_d);
    
    /* Try to send data using the nwk NLDE Data service */
    status = NLDE_DataRequest(
                              appStateMachine.deviceId,
                              gSupportedProfile3_c,
                              vendorId,
                              gAppCmdPayloadLength_c,
                              appCmdPayload,
                              gTxOptions_c
                             );
    /* Exit the state if this is not successful, otherwise wait confirm */
    if(gNWSuccess_c == status)
    {
      appStateMachine.subState = gAppSubStateWaitCnf_c;
    }
    else
    {
      appStateMachine.subState = gAppSubStateEnd_c;
    }
  }
  
  switch(appStateMachine.subState)
  {
    case gAppSubStateWaitCnf_c:
      if((events & gAppEvtMsgFromNlde_c) && 
         (pMsgIn != NULL))
      {
        nwkNldeToAppMsg_t* pNldeMsgIn = (nwkNldeToAppMsg_t*)pMsgIn;
        /* Data confirm received */
        if(pNldeMsgIn->msgType == gNwkNldeDataCnf_c)
        {
          status = pNldeMsgIn->msgData.nwkNldeDataCnf.status;
          appStateMachine.subState = gAppSubStateEnd_c;
        }
      }
      break;
    default:
      break;        
  }
  
  if(appStateMachine.subState == gAppSubStateEnd_c)
  {
    /* Print the status */
    App_PrintResult(status);
    /* Send event to end the state */
    TS_SendEvent(gAppTaskID, gAppEvtStateEnd_c);     
  }     
       
}


/*****************************************************************************
*  App_UnpairState
*
*  Deletes one device from the pair table
*
*****************************************************************************/
static void App_UnpairState(event_t events, void* pMsgIn)
{
  uint8_t status;
  
  if((events & gAppEvtStateStart_c) && (appStateMachine.subState == gAppSubStateStart_c))
  { 
    /* Print a message to the UART */
    UartUtil_Print("\n\rUnpairing device ", gAllowToBlock_d);
    UartUtil_Print(nodeData.pairTableEntry[appStateMachine.deviceId].recipUserString, gAllowToBlock_d);
    UartUtil_Print("... ", gAllowToBlock_d);
    LCD_ClearDisplay();  
    LCD_WriteString(1,"Unpairing ");
    LCD_WriteString(2,"device");     
    /* Try to initiate the nwk Unpair service */
    status = NLME_UnpairRequest(appStateMachine.deviceId);
  
    /* Exit the state if this is not successful, otherwise wait confirm */
    if(gNWSuccess_c == status)
    {
      appStateMachine.subState = gAppSubStateWaitCnf_c;
    }
    else
    {
      appStateMachine.subState = gAppSubStateEnd_c;
    }
  }
  
  switch(appStateMachine.subState)
  {
    case gAppSubStateWaitCnf_c:
      if((events & gAppEvtMsgFromNlme_c) && 
         (pMsgIn != NULL))
      {
        nwkNlmeToAppMsg_t* pNlmeMsgIn = (nwkNlmeToAppMsg_t*)pMsgIn;
        /* Unpair confirm received */
        if(pNlmeMsgIn->msgType == gNwkNlmeUnpairCnf_c)
        {
          status = pNlmeMsgIn->msgData.nwkNlmeUnpairCnf.status;
          appStateMachine.subState = gAppSubStateEnd_c;
          
#if gPBPOrig_d && gFSLProfileRmtPairOrig_d
          if(status != gNWNoMemory_c)
          {
            /* Clear application info about device */
            FLib_MemSet(&appPairedDeviceInfo[pNlmeMsgIn->msgData.nwkNlmeUnpairCnf.deviceId], 0x00, sizeof(appPairedDeviceInfo_t));
            NvSaveOnIdle(gNvDataSet_App_ID_c);
#if gFSLProfilePollOrig_d
            bPollEnable[pNlmeMsgIn->msgData.nwkNlmeUnpairCnf.deviceId] = TRUE;
#endif /* gFSLProfilePollOrig_d */
          }
#endif /* gPBPOrig_d && gFSLProfileRmtPairOrig_d */
        }
      }
      break;
    default:
      break;        
  }
  
  if(appStateMachine.subState == gAppSubStateEnd_c)
  {
    /* Print the status */
    App_PrintResult(status);
      
    /* Send event to end the state */
    TS_SendEvent(gAppTaskID, gAppEvtStateEnd_c);     
  }
}


/*****************************************************************************
*  App_UnpairResponse - Sends an unpair response to the network
*
*****************************************************************************/
static void App_UnpairResponse(void)
{
  uint8_t status;
  /* Send an unpair response only if an unpair indication was received */
  if(appStateMachine.deviceId != gInvalidDeviceId_c)
  {          
    /* Print a message to the UART */
    UartUtil_Print("\n\rDeleting device ", gAllowToBlock_d);
    UartUtil_Print(nodeData.pairTableEntry[appStateMachine.deviceId].recipUserString, gAllowToBlock_d);
    UartUtil_Print(" from pair table... ", gAllowToBlock_d);
    LCD_ClearDisplay();  
    LCD_WriteString(1,"Deleting");
    LCD_WriteString(2," device");          
    /* Send a unpair response and print the status */        
    status = NLME_UnpairResponse(appStateMachine.deviceId);
    
    App_PrintResult(status);
    
#if gPBPOrig_d && gFSLProfileRmtPairOrig_d
    if(status == gNWSuccess_c)
    {
      /* Clear application info about device */
       FLib_MemSet(&appPairedDeviceInfo[appStateMachine.deviceId], 0x00, sizeof(appPairedDeviceInfo_t));
       NvSaveOnIdle(gNvDataSet_App_ID_c);
#if gFSLProfilePollOrig_d
       bPollEnable[appStateMachine.deviceId] = TRUE;
#endif /* gFSLProfilePollOrig_d */
    }
#endif /* gPBPOrig_d && gFSLProfileRmtPairOrig_d */
    
    appStateMachine.deviceId = gInvalidDeviceId_c; 
  }
}


/*****************************************************************************
*  App_PrintPairedDevices
*
*  Used for printing the paired devices on a serial Hyperterminal
*
*****************************************************************************/
static void App_PrintPairedDevices(void)
{ 
  uint8_t iCount;
  uint8_t strNum[] = {'x','x','.','\0'};
  
  /* Print all paired devices in the pair table */
  for(iCount = 0; iCount< gMaxPairingTableEntries_c; iCount++)
  {
    if((iCount + 1) / 10)
    {
      strNum[0] = '0' + ((iCount + 1) / 10);
    }
    else
    {
      strNum[0] = ' ';
    }
    strNum[1] = '0' + ((iCount + 1) % 10);
    
    UartUtil_Print(strNum, gAllowToBlock_d);
    /* recipChannel field will always be different from 0 for an entry that contains valid pair information */
    if(nodeData.pairTableEntry[iCount].recipChannel != 0)
      UartUtil_Print(nodeData.pairTableEntry[iCount].recipUserString, gAllowToBlock_d);
    else
      UartUtil_Print("None", gAllowToBlock_d);
    UartUtil_Print("\n\r", gAllowToBlock_d);       
  }
}



/*****************************************************************************
*  App_HandleDataInd
*
*  Handles data received from paired devices
*
*****************************************************************************/
static void App_HandleDataInd(nwkNldeDataInd_t* pNwkNldeDataInd)
{

  UartUtil_Print("\n\rData received from ", gAllowToBlock_d);
  UartUtil_Print(nodeData.pairTableEntry[pNwkNldeDataInd->deviceId].recipUserString, gAllowToBlock_d);
  
  /* Print information about the received data indication */
  UartUtil_Print("\n\rReceived data parameters:", gAllowToBlock_d);
  /* Specify the payload of data indication, if present */
  UartUtil_Print("\n\r    1.Payload: ", gAllowToBlock_d);  
  if(pNwkNldeDataInd->dataLength > 0)                    
    UartUtil_Tx(&pNwkNldeDataInd->pData[0], pNwkNldeDataInd->dataLength);
  else
    UartUtil_Print("None", gAllowToBlock_d); 
  /* Specify the LQI of the received data */
  UartUtil_Print("\n\r    2.LQI: 0x", gAllowToBlock_d);
  UartUtil_PrintHex(&pNwkNldeDataInd->LQI, 1, gPrtHexSpaces_c);
  /* Specify the Rx options - Data was broadcasted or not */
  UartUtil_Print("\n\r    3.Broadcasted: ", gAllowToBlock_d);
  if(pNwkNldeDataInd->rxFlags & maskRxOptions_Broadcast_c)
    UartUtil_Print("Yes", gAllowToBlock_d);
  else
    UartUtil_Print("No", gAllowToBlock_d);
  /* Specify the Rx options - Data was secured or not */
  UartUtil_Print("\n\r    4.Secured: ", gAllowToBlock_d);
  if(pNwkNldeDataInd->rxFlags & maskRxOptions_UseSecurity_c)
    UartUtil_Print("Yes", gAllowToBlock_d);
  else
    UartUtil_Print("No", gAllowToBlock_d);
  /* Specify profile identifier */  
  UartUtil_Print("\n\r    5.Profile ID: ", gAllowToBlock_d);
  UartUtil_PrintHex(&pNwkNldeDataInd->profileId, 1, 0);
 
  /* Specify the Rx options - Data is vendor specific or not */
  UartUtil_Print("\n\r    6.Vendor specific: ", gAllowToBlock_d);
  if(pNwkNldeDataInd->rxFlags & maskRxOptions_VendorSpecificData_c)
  {
    UartUtil_Print("Yes. Vendor ID: ", gAllowToBlock_d);
    UartUtil_PrintHex(&pNwkNldeDataInd->vendorId[0], 2, 0);
  }
  else
    UartUtil_Print("No", gAllowToBlock_d);
   
  UartUtil_Print("\n\r", gAllowToBlock_d);
}



/*****************************************************************************
*  App_RxEnableRequest
*
*  Prints a message on UART and LCD
*
*****************************************************************************/
static void App_RxEnableRequest(uint8_t mode)
{
  uint8_t   status = gNWDenied_c;
  uint16_t  tmp16Bit;

  switch(mode)
  {
    case 0:
      /* Rx On */
      UartUtil_Print("\n\rSetting Rx to on...", gNoBlock_d);
      status = NLME_RxEnableRequest(0xFFFFFF);
      break;

    case 1:
      /* Rx Off */
      UartUtil_Print("\n\rSetting Rx to off...", gNoBlock_d);
      status = NLME_RxEnableRequest(0x00);
      break;
      
    case 2:
      /* Intermittent Rx */
      UartUtil_Print("\n\rIntermittent Rx...", gNoBlock_d);
      /* Set nwkDutyCycle NIB */
      tmp16Bit = gNwkDutyCycle_c;
      status  |= NLME_SetRequest(gNwkNib_DutyCycle_c, 0, (uint8_t*)&tmp16Bit);
      /* Set the activePeriod and enable intermitentRx mode */
      status = NLME_RxEnableRequest(gNwkActivePeriod_c);
      break;
    default:
      break;  
  }
  
  App_PrintResult(status);  
  
}


/*****************************************************************************
*  App_HandleKeyboard
*
*  Handles all key events for this device.
*
*****************************************************************************/
#ifdef PROCESSOR_MC1323X
  #if (gMC1323xMatrixKBD_d == TRUE)
    static void App_HandleKeyboard(uint8_t events, uint8_t pressedKey)   
    {
      /* Ignore compiler warnings */
      (void)events;
      (void)pressedKey;
  #else
    static void App_HandleKeyboard(key_event_t events)
    {
      /* Ignore compiler warnings */
      (void)events;
  #endif /* gMC1323xMatrixKBD_d */
#else
    static void App_HandleKeyboard(key_event_t events)
    {
      /* Ignore compiler warnings */
      (void)events;
#endif
    if(PWR_CheckIfDeviceCanGoToSleep())
    {        
      UartUtil_Print("\n\rEntering in low power mode is not allowed by the application.",gAllowToBlock_d);
      /* Do not allow device to enter the low power mode. The low power mode is configured in PWR_Configuration.h */
      PWR_DisallowDeviceToSleep();
    }
  }


/*****************************************************************************
* App_UartRxCallBack
*
* This callback is triggered when a new byte is received over the UART
*
*****************************************************************************/
static void App_UartRxCallBack(void)
{
  TS_SendEvent(gAppTaskID, gAppEvtRxFromUart_c);
}


/*****************************************************************************
*  App_RxEnableRequest
*
*  Prints a message on UART and LCD
*
*****************************************************************************/
static void App_PrintResult(uint8_t status)
{
  if(status == gNWSuccess_c)
  {
    UartUtil_Print("Successful\n\r", gAllowToBlock_d);
    LCD_ClearDisplay();  
    LCD_WriteString(1," Successful");   
  }
  else
  {
    UartUtil_Print("Failed\n\r", gAllowToBlock_d);
    LCD_ClearDisplay();  
    LCD_WriteString(1," Failed");
  }
}


#if gFSLProfileTask_d
/*****************************************************************************
*  App_FSLGetSupportedFeaturesState
*
*  Sends a GetSupportedFeatures request to a paired device.
*
*****************************************************************************/
static void App_FSLGetSupportedFeaturesState(event_t events, void* pMsgIn)
{
  uint8_t   status;
  uint8_t*  supportedFeaturesMap = NULL;

  if((events & gAppEvtStateStart_c) && (appStateMachine.subState == gAppSubStateStart_c))
  {
    UartUtil_Print("\n\rGetting FSL private profile supported features... ", gAllowToBlock_d);
    
    /* Try to send data using the nwk NLDE Data service */
    status = FSLProfile_GetSupportedFeatures(
                                appStateMachine.deviceId,
                                gTxOptions_UseSecurity_c
                              );
    /* Exit the state if this is not successful, otherwise wait confirm */
    if(gNWSuccess_c == status)
    {
      appStateMachine.subState = gAppSubStateWaitCnf_c;
    }
    else
    {
      appStateMachine.subState = gAppSubStateEnd_c;
    }
  }
  
  switch(appStateMachine.subState)
  {
    case gAppSubStateWaitCnf_c:
      if((events & gAppEvtMsgFromFSLProfile_c) && 
         (pMsgIn != NULL))
      {
        fslProfileToAppMsg_t* pFSLProfileMsgIn = (fslProfileToAppMsg_t*)pMsgIn;
        /* Data confirm received */
        if(pFSLProfileMsgIn->msgType == gFSLProfileGetSupportedFeaturesCnf_c)
        {
          status                    = pFSLProfileMsgIn->msgData.fslProfileGetSupportedFeaturesCnf.status;
          supportedFeaturesMap      = &pFSLProfileMsgIn->msgData.fslProfileGetSupportedFeaturesCnf.supportedFeaturesMap[0];
          appStateMachine.subState  = gAppSubStateEnd_c;
        }
      }
      break;
      
    default:
      break;        
  }
  
  if(appStateMachine.subState == gAppSubStateEnd_c)
  {
    /* Print the status */
    App_PrintResult(status);
    
    if((status == gNWSuccess_c) && (supportedFeaturesMap != NULL))
    {
      if(FLib_MemCmp(supportedFeaturesMap, 0x00, 4))
        UartUtil_Print("Supported features: None", gAllowToBlock_d);
      else
      {
        if(supportedFeaturesMap[0] & gFSLProfileDeviceSupportFragOrig_c)
          UartUtil_Print("\n\r  * Fragmented data originator", gAllowToBlock_d); 
        if(supportedFeaturesMap[0] & gFSLProfileDeviceSupportFragRecip_c)
          UartUtil_Print("\n\r  * Fragmented data recipient", gAllowToBlock_d); 
        if(supportedFeaturesMap[0] & gFSLProfileDeviceSupportPollOrig_c)
          UartUtil_Print("\n\r  * Poll originator", gAllowToBlock_d); 
        if(supportedFeaturesMap[0] & gFSLProfileDeviceSupportPollRecip_c)
          UartUtil_Print("\n\r  * Poll recipient", gAllowToBlock_d); 
        if(supportedFeaturesMap[0] & gFSLProfileDeviceSupportRmtPairOrig_c)
          UartUtil_Print("\n\r  * Remote pair originator", gAllowToBlock_d); 
        if(supportedFeaturesMap[0] & gFSLProfileDeviceSupportRmtPairRecip_c)
          UartUtil_Print("\n\r  * Remote pair recipient", gAllowToBlock_d);
        if(supportedFeaturesMap[0] & gFSLProfileDeviceSupportMenuBrowser_c)
          UartUtil_Print("\n\r  * Over the air menus browser", gAllowToBlock_d); 
        if(supportedFeaturesMap[0] & gFSLProfileDeviceSupportMenuOwner_c)
          UartUtil_Print("\n\r  * Over the air menus owner", gAllowToBlock_d); 
        if(supportedFeaturesMap[1] & (gFSLProfileDeviceSupportMenuDisplayer_c % 0xFF))
          UartUtil_Print("\n\r  * Over the air menus displayer", gAllowToBlock_d); 
      }
      UartUtil_Print("\n\r", gAllowToBlock_d);
    }
        
    /* Send event to end the state */
    TS_SendEvent(gAppTaskID, gAppEvtStateEnd_c);     
  }     
       
}
#endif /* gFSLProfileTask_d */


#if gFSLProfileFragTxOrig_d
/*****************************************************************************
*  App_FSLFragTxState
*
*  Sends a FSL Profile FragTx request
*
*****************************************************************************/
static void App_FSLFragTxState(event_t events, void* pMsgIn)
{
  uint8_t   status;
  uint8_t*  fragTxDataPayload = &fragTxRxDataBuffer[0];

  if((events & gAppEvtStateStart_c) && (appStateMachine.subState == gAppSubStateStart_c))
  {
    /* Set FragTxRx buffer state to BusyApp */
    status = FSLProfile_SetFragTxRxBufferStateRequest(gFragTxRxBufferBusyApp_c);
    
    if(status == gNWSuccess_c)
    {
      /* Write data in FragTx buffer */
      App_FSLWriteDataInFragTxBuffer();
   
      UartUtil_Print("\n\rSending FragTx data... ", gAllowToBlock_d);
      
      /* Try to send data using the FSL Profile FragTx service */
      status = FSLProfile_FragTxRequest(
                                appStateMachine.deviceId,
                                gAppFSLFragTxPayloadLength_c,
                                fragTxDataPayload,
                                gTxOptions_UseSecurity_c
                               );
    }
    
    /* Exit the state if this is not successful, otherwise wait confirm */
    if(gNWSuccess_c == status)
    {
      appStateMachine.subState = gAppSubStateWaitCnf_c;
    }
    else
    {
      appStateMachine.subState = gAppSubStateEnd_c;
    }
  }
  
  switch(appStateMachine.subState)
  {
    case gAppSubStateWaitCnf_c:
      if((events & gAppEvtMsgFromFSLProfile_c) && 
         (pMsgIn != NULL))
      {
        fslProfileToAppMsg_t* pFSLProfileMsgIn = (fslProfileToAppMsg_t*)pMsgIn;
        /* FragTx confirm received */
        if(pFSLProfileMsgIn->msgType == gFSLProfileFragCnf_c)
        {
          status = pFSLProfileMsgIn->msgData.fslProfileFragCnf.status;
          appStateMachine.subState = gAppSubStateEnd_c;
        }
      }
      break;
    default:
      break;        
  }
  
  if(appStateMachine.subState == gAppSubStateEnd_c)
  {    
    /* Print the status */
    App_PrintResult(status);
    /* Set FragTx buffer state to free */
    UartUtil_Print("\n\rSet FragTxRx buffer state to Free... ", gAllowToBlock_d); 
    App_PrintResult(FSLProfile_SetFragTxRxBufferStateRequest(gFragTxRxBufferFree_c));
    /* Send event to end the state */
    TS_SendEvent(gAppTaskID, gAppEvtStateEnd_c);     
  }           
}


/*****************************************************************************
*  App_FSLWriteDataInFragTxBuffer
*
*  Writes data in the FragTx buffer
*
*****************************************************************************/
static void App_FSLWriteDataInFragTxBuffer(void)
{
  uint16_t iCount;
  
  for(iCount = 0; iCount < gFSLProfileFragTxRxBufferLength_c; iCount++)
    fragTxRxDataBuffer[iCount] = 'a' + iCount % 22;     
}


#endif /* gFSLProfileFragTxOrig_d */


#if gFSLProfileRmtPairOrig_d
/*****************************************************************************
*  App_FSLRmtPairState
*
*  Makes a remote pair request to FSL profile
*
*****************************************************************************/
static void App_FSLRmtPairState(event_t events, void* pMsgIn)
{
  uint8_t status;
        
  if((events & gAppEvtStateStart_c) && (appStateMachine.subState == gAppSubStateStart_c))
  {
    UartUtil_Print("\n\rRemote pair (first two devices from pair table)... ", gAllowToBlock_d);
    
   /* Make a remote pair request.*/
    status = FSLProfile_RmtPairRequest(
                                    gAppFSLRmtPairRespTimeout_c, 
                                    0, 
                                    appPairedDeviceInfo[0].recipAppCapabilities,
                                    appPairedDeviceInfo[0].recipDeviceTypeList,
                                    appPairedDeviceInfo[0].recipProfilesList,
                                    1,
                                    appPairedDeviceInfo[1].recipAppCapabilities,
                                    appPairedDeviceInfo[1].recipDeviceTypeList,
                                    appPairedDeviceInfo[1].recipProfilesList
                                  ); 
    
    /* Exit the state if this is not successful, otherwise wait confirm */
    if(gNWSuccess_c == status)
    {
      appStateMachine.subState = gAppSubStateWaitCnf_c;
    }
    else
    {
      appStateMachine.subState = gAppSubStateEnd_c;
    }
  }
  
  switch(appStateMachine.subState)
  {
    case gAppSubStateWaitCnf_c:
      if((events & gAppEvtMsgFromFSLProfile_c) && 
         (pMsgIn != NULL))
      {
        fslProfileToAppMsg_t* pFSLProfileMsgIn = (fslProfileToAppMsg_t*)pMsgIn;
        
        /* Remote pair confirm received */
        if(pFSLProfileMsgIn->msgType == gFSLProfileRmtPairCnf_c)
        {
          status = pFSLProfileMsgIn->msgData.fslProfileRmtPairCnf.status;
          appStateMachine.subState = gAppSubStateEnd_c;
        }
      }
      break;
    default:
      break;        
  }
  
  if(appStateMachine.subState == gAppSubStateEnd_c)
  {
    /* Print the status */
    App_PrintResult(status);
    
    /* Send event to end the state */
    TS_SendEvent(gAppTaskID, gAppEvtStateEnd_c);     
  }     
       
}
#endif /* gFSLProfileRmtPairOrig_d */


#if gFSLProfilePollOrig_d
/*****************************************************************************
*  App_FSLPoll - Poll or stop polling a requested device
*
*****************************************************************************/
static void App_FSLPoll(uint8_t deviceId)
{
  uint8_t status;
  
  if(bPollEnable[deviceId] == TRUE)
  {
    /* Enable poll for selected device */
    UartUtil_Print("\n\rStart poll ... ", gAllowToBlock_d);
  }
  else
  {
    /* Disable poll for selected device */
    UartUtil_Print("\n\rStop poll ... ", gAllowToBlock_d);
  }

  /* Make the poll request */
  status = FSLProfile_PollRequest(deviceId, bPollEnable[deviceId]);
          
  if (status == gNWSuccess_c)
  {
    /* Change poll enable info for the selected device */
    bPollEnable[deviceId] = !bPollEnable[deviceId];
  }
   
  /* Print the status */
  App_PrintResult(status);
}
#endif /* gFSLProfilePollOrig_d */


#if gPBPOrig_d
/*****************************************************************************
*  App_PushButtonPairingState
*
*  Handles the push button pairing process
*
*****************************************************************************/
static void App_PushButtonPairingState(event_t events, void* pMsgIn)
{
  uint8_t status;
  pushButtonToAppMsg_t* pPBPMsgIn = (pushButtonToAppMsg_t*)pMsgIn;
              
  if((events & gAppEvtStateStart_c) && (appStateMachine.subState == gAppSubStateStart_c))
  {
    
    UartUtil_Print("\n\rPush Button Pairing... ", gAllowToBlock_d);
    
   /* Make a push button pairing request.*/
    status = PBP_PushButtonPairOrigRequest(
                                    gAppZRCDiscoverySearchedPanId_c, 
                                    gAppZRCDiscoverySearchedShortAddress_c, 
                                    gAppZRCDiscoverySearchedDeviceType_c, 
                                    localAppCapabilities,
                                    localDeviceTypesList,
                                    localProfilesList,
                                    localAppCapabilities.nrSupportedProfiles,
                                    localProfilesList,
                                    gAppZRCKeyExTransferCount_c,
                                    gAppZRCRequestAppAcceptToPair_c,
                                    gAppZRCTimeToWaitAppAcceptToPair_c
                                  ); 
    
    /* Exit the state if this is not successful, otherwise wait confirm */
    if(gNWSuccess_c == status)
    {
      appStateMachine.subState = gAppSubStateWaitCnf_c;
    }
    else
    {
      appStateMachine.subState = gAppSubStateEnd_c;
    }
  }
  
  switch(appStateMachine.subState)
  {
    case gAppSubStateWaitCnf_c:
      if((events & gAppEvtMsgFromPushButton_c) && 
         (pMsgIn != NULL))
      {
        /* This substate handles both the messages:
            -  gPushButtonPairOrigCnf_c - informing the application that
        the push button pair process has completed 
            - gPushButtonPairOrigContinueInd_c - asking for application's aproval
        to continue the push button pair procedure by pairing with the just discovered device */
        if(pPBPMsgIn->msgType == gPushButtonPairOrigCnf_c)
        {
          status = pPBPMsgIn->msgData.pushButtonPairOrigCnf.status;
          appStateMachine.subState = gAppSubStateEnd_c;
        }
        else if(pPBPMsgIn->msgType == gPushButtonPairOrigContinueInd_c)
        {
          /* The application can decide here whether it wants to continue the push button pair
          process or not, after consulting the information about the recipient provided by the profile
          in the pushButtonPairOrigContinueInd_t message.
             This demo app always choose to continue with the pair process */
          (void)PBP_PushButtonPairOrigContinueResponse(TRUE);
        }
      }
      break;

    default:
      break;        
  }
  
  if(appStateMachine.subState == gAppSubStateEnd_c)
  {
    
    /* Print the status */
    App_PrintResult(status);
    
    if(status == gNWSuccess_c)
    {
      pushButtonPairOrigCnf_t* pPBPOrigCnf = &pPBPMsgIn->msgData.pushButtonPairOrigCnf;
#if gFSLProfileRmtPairOrig_d
      appPairedDeviceInfo_t*              currentPairedDeviceInfo           = &appPairedDeviceInfo[pPBPOrigCnf->deviceId];
#endif /* gFSLProfileRmtPairOrig_d */
            
      UartUtil_Print("Successfully paired with device ", gAllowToBlock_d);
      UartUtil_Print(pPBPOrigCnf->recipUserString, gAllowToBlock_d);  
      UartUtil_Print(".\n\r", gAllowToBlock_d);
      
#if gFSLProfileRmtPairOrig_d
      /* Save application info about paired device */
      *(uint8_t*)&currentPairedDeviceInfo->recipAppCapabilities = *(uint8_t*)&pPBPOrigCnf->recipAppCapabilities;
      FLib_MemCpy(currentPairedDeviceInfo->recipDeviceTypeList, pPBPOrigCnf->recipDeviceTypeList, pPBPOrigCnf->recipAppCapabilities.nrSupportedDeviceTypes);
      FLib_MemCpy(currentPairedDeviceInfo->recipProfilesList, pPBPOrigCnf->recipProfilesList, pPBPOrigCnf->recipAppCapabilities.nrSupportedProfiles);                  
      NvSaveOnIdle(gNvDataSet_App_ID_c);
#endif /* gFSLProfileRmtPairOrig_d */
    }
    
    /* Send event to end the state */
    TS_SendEvent(gAppTaskID, gAppEvtStateEnd_c);     
  }     
       
}
#endif /* gPBPOrig_d */


#if gZRCProfileCommandTxRx_d
/*****************************************************************************
*  App_ZRCSendCommandState
*
*  Sends a ZRC comand to a selected device.
*
*****************************************************************************/
static void App_ZRCSendCommandState(event_t events, void* pMsgIn)
{
  uint8_t status;
  zrcProfileToAppMsg_t* pZRCProfileMsgIn = (zrcProfileToAppMsg_t*)pMsgIn;

  if((events & gAppEvtStateStart_c) && (appStateMachine.subState == gAppSubStateStart_c))
  { 
    uint8_t appZRCCmdPayload[]   = gAppZRCCmdPayload_c;
    uint8_t vendorId[]            = gDefaultValueOfVendorId_c;
    uint8_t payloadLength         = gAppZRCCmdPayloadLength_c;
    uint8_t txOptions             = gTxOptions_c;

    UartUtil_Print("\n\rSending ZRC request... ", gAllowToBlock_d);
    
    /* The Discovery Request has no payload (appZRCCmdPayload is not used) */
    if(appStateMachine.zrcCmdCode == gZRC_CmdCode_DiscoveryRequest_c) 
    {
      payloadLength = 0x00;        /* set the payload to be zero */
      /* Discovery command is a specific ZRC command*/
      txOptions &= ~maskTxOptions_VendorSpecificData_c;
    }
    
    
    /* Try to send a command using the ZRC profile service */
    status = ZRCProfile_CommandRequest(
                              appStateMachine.deviceId,
                              appStateMachine.zrcCmdCode,
                              gAppZRCCmd_c,
                              vendorId,
                              payloadLength,
                              appZRCCmdPayload,
                              txOptions
                             );
                             
    /* Send the released command, so that not to start repeating the pressed command
       (if previous was sent gZRC_CmdCode_UserCtrlPressedAndRepeat_c command).*/ 
    if((status == gNWSuccess_c) &&
      (appStateMachine.zrcCmdCode == gZRC_CmdCode_UserCtrlPressedAndRepeat_c) &&
      ((txOptions & maskTxOptions_VendorSpecificData_c) != maskTxOptions_VendorSpecificData_c)) 
    {
      status = ZRCProfile_CommandRequest(
                                appStateMachine.deviceId,
                                gZRC_CmdCode_UserCtrlReleased_c,
                                gAppZRCCmd_c,
                                vendorId,
                                0x00,
                                NULL,
                                txOptions
                               );      
     }
     
    /* Exit the state if this is not successful, otherwise wait confirm */
    if(gNWSuccess_c == status)
    {
      appStateMachine.subState = gAppSubStateWaitCnf_c;
    }
    else
    {
      appStateMachine.subState = gAppSubStateEnd_c;
    }
  }
  
  switch(appStateMachine.subState)
  {
    case gAppSubStateWaitCnf_c:
      if((events & gAppEvtMsgFromZRCProfile_c) && 
         (pMsgIn != NULL))
      {
        /* ZRC command confirm received */
        if((pZRCProfileMsgIn->msgType == gZRCProfileCommandCnf_c) ||
           (pZRCProfileMsgIn->msgType == gZRCProfileDiscoveryCmdCnf))
        {
         /* First byte from the confirm message is the status */
          status = ((uint8_t*)&pZRCProfileMsgIn->msgData)[0];
          appStateMachine.subState = gAppSubStateEnd_c;
        }
      }
      break;
    default:
      break;        
  }
  
  if(appStateMachine.subState == gAppSubStateEnd_c)
  {
    /* Print the status */
    App_PrintResult(status);

    /* Print the supported commands */
    if((pZRCProfileMsgIn->msgType == gZRCProfileDiscoveryCmdCnf) &&
       (pZRCProfileMsgIn->msgData.zrcProfileDiscoveryCmdCnf.status == gNWSuccess_c) ) 
    {
     uint8_t supportedCmdByte;
     uint8_t shiftBits, i, haveCmdsFlag = FALSE;
      
     UartUtil_Print("\n\r Commands supported by the remote node: \n\r", gAllowToBlock_d);
     for(i = 0; i < gCmdsSupportedFieldLength_c; i++) 
     {
      supportedCmdByte = pZRCProfileMsgIn->msgData.zrcProfileDiscoveryCmdCnf.cmdSupportedBitMap[i];
      shiftBits = 0;
      
      while(supportedCmdByte) 
      {
        if( supportedCmdByte & 0x01) 
        {  
         /* Find the command Id */
         uint8_t cmdId = ((i >> 1) << 4) | (shiftBits + (8*(i%2))) ;
         UartUtil_Print("0x", gAllowToBlock_d);
         UartUtil_PrintHex(&cmdId, 1, gPrtHexSpaces_c|gPrtHexCommas_c);
         haveCmdsFlag = TRUE;
        }
        shiftBits++; 
        supportedCmdByte = (supportedCmdByte >> 0x01);
      } /* end while */
     } /* end for */
     
     if(!haveCmdsFlag) {      
       UartUtil_Print(" NO ZRC profile commands are supported. \n\r", gAllowToBlock_d);
     }
    }
    
    /* Send event to end the state */
    TS_SendEvent(gAppTaskID, gAppEvtStateEnd_c);     
  }     
       
}


/*****************************************************************************
*  App_ZRCHandleCommandInd
*
*  Handles the commands received on ZRC profile from paired devices
*
*****************************************************************************/
static void App_ZRCHandleCommandInd(zrcProfileCommandInd_t* pZRCProfileCommandInd)
{

  if(pZRCProfileCommandInd->rxFlags & maskRxOptions_VendorSpecificData_c)
  {
    UartUtil_Print("\n\rZRC vendor specific data received from ", gAllowToBlock_d);
  }
  else
  {    
    /* Handle the ZRC profile commands */
    switch(pZRCProfileCommandInd->commandCode)
    {
      case gZRC_CmdCode_UserCtrlPressed_c:
        UartUtil_Print("\n\rUser control pressed:", gAllowToBlock_d);
        switch(pZRCProfileCommandInd->command)
        {
          case gZRC_CmdRC_Select_c:
            UartUtil_Print("'Select'", gAllowToBlock_d);
            break;

          case gZRC_CmdRC_Up_c:
            UartUtil_Print("'Up'", gAllowToBlock_d);
            break;

          case gZRC_CmdRC_Down_c:
            UartUtil_Print("'Down'", gAllowToBlock_d);
            break;

          case gZRC_CmdRC_Left_c:
            UartUtil_Print("'Left'", gAllowToBlock_d);
            break;

          case gZRC_CmdRC_Right_c:
            UartUtil_Print("'Rigth'", gAllowToBlock_d);
            break;

          
          /* ...
             Application should implement here the code for handling the supported ZRC commands
             ... */
          

          default:
            UartUtil_PrintHex(&pZRCProfileCommandInd->command, 1, gPrtHexSpaces_c);
          
        }
        break;
              
      case gZRC_CmdCode_UserCtrlRepeated_c:
        UartUtil_Print("\n\rUser control repetead", gAllowToBlock_d);
        break;

      case gZRC_CmdCode_UserCtrlReleased_c:
        UartUtil_Print("\n\rUser control released", gAllowToBlock_d);
        break;
        
     default:
        UartUtil_PrintHex(&pZRCProfileCommandInd->commandCode, 1, gPrtHexSpaces_c);
        break;
      
    }
    UartUtil_Print("\n\rZRC command received from ", gAllowToBlock_d);
  }
  
  if(pZRCProfileCommandInd->deviceId != 0xFF)
    UartUtil_Print(nodeData.pairTableEntry[pZRCProfileCommandInd->deviceId].recipUserString, gAllowToBlock_d);
  else
    UartUtil_Print("a not paired device", gAllowToBlock_d);
  
  /* Print information about the received ZRC command */
  UartUtil_Print("\n\rReceived command parameters:", gAllowToBlock_d);
  /* Specify the payload of the command, if present */
  UartUtil_Print("\n\r    1.Payload: ", gAllowToBlock_d);
  
  if(pZRCProfileCommandInd->dataLength > 0)                    
    UartUtil_Tx(&pZRCProfileCommandInd->pData[0], pZRCProfileCommandInd->dataLength);
  else
    UartUtil_Print("None", gAllowToBlock_d);

  
  /* Specify the LQI the ZRC command was received with */
  UartUtil_Print("\n\r    2.LQI: 0x", gAllowToBlock_d);
  UartUtil_PrintHex(&pZRCProfileCommandInd->LQI, 1, gPrtHexSpaces_c);
  /* Specify the Rx options - Command was broadcasted or not */
  UartUtil_Print("\n\r    3.Broadcasted: ", gAllowToBlock_d);
  if(pZRCProfileCommandInd->rxFlags & maskRxOptions_Broadcast_c)
    UartUtil_Print("Yes", gAllowToBlock_d);
  else
    UartUtil_Print("No", gAllowToBlock_d);
  /* Specify the Rx options - ZRC command was secured or not */
  UartUtil_Print("\n\r    4.Secured: ", gAllowToBlock_d);
  if(pZRCProfileCommandInd->rxFlags & maskRxOptions_UseSecurity_c)
    UartUtil_Print("Yes", gAllowToBlock_d);
  else
    UartUtil_Print("No", gAllowToBlock_d);
  /* Specify the Rx options - ZRC command is vendor specific or not */
  UartUtil_Print("\n\r    5.Vendor specific: ", gAllowToBlock_d);
  if(pZRCProfileCommandInd->rxFlags & maskRxOptions_VendorSpecificData_c)
  {
    UartUtil_Print("Yes. Vendor ID: ", gAllowToBlock_d);
    UartUtil_PrintHex(&pZRCProfileCommandInd->vendorId[0], 2, 0);
  }
  else
    UartUtil_Print("No", gAllowToBlock_d);
  
  UartUtil_Print("\n\r", gAllowToBlock_d);
}
#endif /* gZRCProfileCommandTxRx_d */


#if gFSLProfileTask_d
/*****************************************************************************
*  App_MessageIsForFSLProfile
*
*  Returns TRUE if a message is for the FSL Profile and FALSE otherwise
*
*****************************************************************************/
static bool_t App_MessageIsForFSLProfile(nwkNldeToAppMsg_t* nwkNldeToAppMsg)
{
  
  if(nwkNldeToAppMsg->msgType == gNwkNldeDataInd_c)
  {
    uint8_t vendorId[2] = {gFSLVendorId_c};
    
    if((nwkNldeToAppMsg->msgData.nwkNldeDataInd.profileId == gFSLProfileId_c)                   &&
       (nwkNldeToAppMsg->msgData.nwkNldeDataInd.rxFlags & maskRxOptions_VendorSpecificData_c) &&
       (FLib_MemCmp(nwkNldeToAppMsg->msgData.nwkNldeDataInd.vendorId, vendorId, 2)))
    {
      /* NLDE indication message is for FSL profile */
      return TRUE;   
    }
  }
  else if(nwkNldeToAppMsg->msgData.nwkNldeDataCnf.profileId == gFSLProfileId_c)
  {
    /* NLDE confirm message is for FSL profile */
    return TRUE;
  }
 
  /* NLDE message is not for FSL profile */
  return FALSE;      
}
#endif /* gFSLProfileTask_d */


/************************************************************************************
* NLME Sap Handler. This function will be called by the NW layer with a pointer to  
* nwkNlmeToAppMsg_t.
* The function implementation in the application should queue the the message and send
* an event to the application task.
*   
* Return value:
*    None
* 
************************************************************************************/
void NWK_NLME_SapHandler(nwkNlmeToAppMsg_t* nwkNlmeToAppMsg)
{
#if gPBPTask_d
  if(appStateMachine.state == gAppStatePushButtonPairing_c)
  {
    /* NLME message is for Push Button Pair */
    PBP_HandleNwkNlmeMsg(nwkNlmeToAppMsg);
  }
  else
#endif /* gZRCProfileCommandTxRx_d */
  {
    /* Put the incoming NLME message in the applications input queue. */
    MSG_Queue(&mNlmeAppInputQueue, nwkNlmeToAppMsg);  
    TS_SendEvent(gAppTaskID, gAppEvtMsgFromNlme_c);
  }
}

/************************************************************************************
* NLDE Sap Handler. This function will be called by the NW layer with a pointer to  
* nwkNldeToAppMsg_t.
* The function implementation in the application should queue the the message and send
* an event to the application task.
*   
* Return value:
*    None
* 
************************************************************************************/
void NWK_NLDE_SapHandler(nwkNldeToAppMsg_t* nwkNldeToAppMsg)
{
  /* Put the incoming NLDE message in the right queue. */
#if gFSLProfileTask_d
  if(App_MessageIsForFSLProfile(nwkNldeToAppMsg))
  {
    /* NLDE message is for FSL profile */
    FSLProfile_HandleNwkNldeMsg(nwkNldeToAppMsg);
  }
  else
#endif /* gFSLProfileTask_d */
  {
#if gZRCProfileCommandTxRx_d
    if(((nwkNldeToAppMsg->msgType == gNwkNldeDataInd_c) && (nwkNldeToAppMsg->msgData.nwkNldeDataInd.profileId == gZRCProfileId_c)) ||
       ((nwkNldeToAppMsg->msgType == gNwkNldeDataCnf_c) && (nwkNldeToAppMsg->msgData.nwkNldeDataCnf.profileId == gZRCProfileId_c)))
    {
      /* NLDE message is for ZRC profile */
      ZRCProfile_HandleNwkNldeMsg(nwkNldeToAppMsg);
    }
    else
#endif /* gZRCProfileCommandTxRx_d */
    {     
      /* Put the incoming NLDE message in the applications input queue. */
      MSG_Queue(&mNldeAppInputQueue, nwkNldeToAppMsg);  
      TS_SendEvent(gAppTaskID, gAppEvtMsgFromNlde_c);
    }
  }
}


#if gFSLProfileTask_d
/************************************************************************************
* FSLProfile Sap Handler. This function will be called by the FSL Profile layer with a 
* pointer to FSLProfileToAppMsg_t.
* The function implementation in the application should queue the the message and send
* an event to the application task.
*   
* Return value:
*    None
* 
************************************************************************************/
void FSLProfile_App_SapHandler(fslProfileToAppMsg_t* fslProfileToAppMsg)
{
  /* Put the incoming FSL profile message in the applications input queue. */
  MSG_Queue(&mFSLProfileAppInputQueue, fslProfileToAppMsg);  
  TS_SendEvent(gAppTaskID, gAppEvtMsgFromFSLProfile_c);
}
#endif /* gFSLProfileTask_d */

#if gZRCProfileCommandTxRx_d
/************************************************************************************
* ZRCProfile Sap Handler. This function will be called by the ZRC Profile layer with a 
* pointer to zrcProfileToAppMsg_t.
* The function implementation in the application should queue the the message and send
* an event to the application task.
*   
* Return value:
*    None
* 
************************************************************************************/
void ZRCProfile_App_SapHandler(zrcProfileToAppMsg_t* zrcProfileToAppMsg)
{
  /* Put the incoming ZRC profile message in the applications input queue. */
  MSG_Queue(&mZRCProfileAppInputQueue, zrcProfileToAppMsg);  
  TS_SendEvent(gAppTaskID, gAppEvtMsgFromZRCProfile_c);
}
#endif /* gZRCProfileCommandTxRx_d */


#if gPBPTask_d
/************************************************************************************
* PBP Sap Handler. This function will be called by the PBP layer with a 
* pointer to pushButtonToAppMsg_t.
* The function implementation in the application should queue the the message and send
* an event to the application task.
*   
* Return value:
*    None
* 
************************************************************************************/
void PBP_APP_SapHandler(pushButtonToAppMsg_t* pbpToAppMsg)
{
  MSG_Queue(&mPushButtonAppInputQueue, pbpToAppMsg);  
  TS_SendEvent(gAppTaskID, gAppEvtMsgFromPushButton_c);  
}
#endif

